Deployment checklist — FortiPass Chrome extension

This file describes the recommended steps to prepare, configure, and deploy your extension and Firebase backend for production.

Prerequisites
- Node.js (LTS) and npm
- A Google Cloud project (for OAuth client ID)
- A Firebase project (linked to the Google Cloud project) with Firestore & Authentication enabled
- Your Chrome extension source (this workspace)

Summary of steps
1. Prepare Firebase project and enable features
2. Create OAuth 2.0 Client ID (Web application) and register extension redirect URI
3. Update `manifest.json` with production `oauth2.client_id`
4. Ensure `firebase-init.js` contains correct Firebase configuration for the production project
5. Configure Firestore rules and deploy them
6. (Optional) Deploy a small hosting site (privacy, landing page)
7. Pack and publish the extension to Chrome Web Store

Detailed instructions

1) Create/choose Firebase + Google Cloud project
- In Google Cloud Console, create or pick a project.
- In Firebase Console, create a Firebase project and link it to the same Google Cloud project.

2) Enable Firebase Authentication and add authorized domain
- In Firebase Console > Authentication > Sign-in method: enable "Google" provider.
- In Firebase Console > Authentication > Settings > Authorized domains, add the extension redirect domain (see next step to get exact string).

3) Get your extension's redirect URI and extension ID (development)
- Load your unpacked extension in Chrome (chrome://extensions/ → Developer mode → Load unpacked).
- In the extension popup console (or background), run these in DevTools to get exact values:

```js
console.log('extension id =>', chrome.runtime.id);
console.log('redirect uri =>', chrome.identity.getRedirectURL());
```

- The redirect URI looks like: `https://<EXTENSION_ID>.chromiumapp.org/` (include trailing slash). Use that exact value in the OAuth client.

4) Create OAuth client in Google Cloud
- In Google Cloud Console > APIs & Services > Credentials > Create Credentials > OAuth client ID.
- Choose "Web application".
- Under "Authorized redirect URIs", add the exact redirect URI from step 3.
- Create the client and copy the generated `Client ID`.

5) Update your extension manifest
- Open `manifest.json` and set `oauth2.client_id` to the new Client ID.

Example:
```json
"oauth2": {
  "client_id": "YOUR_CLIENT_ID.apps.googleusercontent.com",
  "scopes": [
    "https://www.googleapis.com/auth/userinfo.email",
    "https://www.googleapis.com/auth/userinfo.profile"
  ]
}
```

6) Configure Firebase config in `firebase-init.js`
- Confirm the `firebaseConfig` object matches the Firebase project values (apiKey, projectId, etc.).
- You can copy the config from Firebase Console > Project settings > General.

7) Firestore rules (basic example)
- We provide a sample `firestore.rules` which allows read/write to `/credentials/{uid}` only for authenticated users matching the document id.
- Review and harden rules before production.

8) Deploy Firestore rules and hosting
- Install Firebase CLI or use npx.

PowerShell commands (run from project root):

```powershell
# Install dependencies locally
npm init -y
npm install --save-dev firebase-tools

# Login to Firebase
npx firebase login

# Associate your local project with the Firebase project ID
npx firebase use --add

# Deploy Firestore rules and hosting
npx firebase deploy --only firestore:rules,hosting
```

Notes:
- `firebase.json` and `.firebaserc` are included in this repo as templates.
- If you don't need hosting, you can skip hosting in the deploy command.

9) Packing & publishing extension
- Chrome Web Store requires a packed, signed extension (.zip or crx via Developer Dashboard).
- In chrome://extensions, use "Pack extension" (choose private key to keep extension ID stable). Alternatively, use the Chrome Web Store developer API to upload.
- When publishing publicly, ensure the OAuth client in Google Cloud has the production redirect URI (packaging changes extension id) — register the production extension ID redirect URI.

10) Clearing cached tokens for testing
- If you change client IDs or OAuth setup, clear cached tokens in the extension popup console:

```js
chrome.identity.getAuthToken({interactive:false}, (token) => {
  if (token) chrome.identity.removeCachedAuthToken({token}, () => console.log('old token cleared'));
});
```

Security checklist before publishing
- Remove any debug console logs (or minimize them)
- Remove hard-coded debug/test client IDs or secret values
- Confirm Firestore rules are restrictive
- Confirm content security policy in `manifest.json` is tight and only allows required origins

Support & next steps I can do for you
- Create a small hosting page (privacy policy) and deploy it to Firebase Hosting
- Move auth logic to background service worker for robustness
- Help pack and upload to Chrome Web Store

If you want, I can now:
- (A) Add template `firebase.json`, `.firebaserc`, and `firestore.rules` to your project (I will), and a `deploy.ps1` script for Windows PowerShell with the commands above.
- (B) Walk you step-by-step through creating the OAuth client in Google Cloud with screenshots/commands.

Tell me which you'd like next (A or B).